const express = require('express');
const cors = require('cors');
const crypto = require('crypto');

const app = express();
const PORT = 3001;

// 미들웨어
app.use(cors());
app.use(express.json());
app.use(express.static('public'));

app.get('/', (req, res) => {
  res.sendFile(__dirname + '/index.html');
});


app.get('/v2-examples', (req, res) => {
  res.sendFile(__dirname + '/v2-examples.html');
});

app.get('/v2-examples.html', (req, res) => {
  res.sendFile(__dirname + '/v2-examples.html');
});

// usage-examples.html 서빙
app.get('/usage-examples', (req, res) => {
  res.sendFile(__dirname + '/usage-examples.html');
});

// usage-examples.html 직접 접근
app.get('/usage-examples.html', (req, res) => {
  res.sendFile(__dirname + '/usage-examples.html');
});

// 파트너 정보 (실제로는 데이터베이스에서 관리)
const PARTNER_CONFIG = {
  partnerId: 1,
  apiKey: 'pk_8dc1b3e4a5754daebdac9b174f133e1f', 
  apiSecret: 'sk_ff55ffdb540c4db995b8cd853038a446'
};

// HMAC 서명 생성 (Widget API 방식에 맞춤)
function generateSignature(method, path, timestamp, secret) {
    // 서명 데이터 구성: METHOD + PATH + TIMESTAMP
  const message = method + path + timestamp;
  
  console.log('📝 서명 생성 메시지:', message);
  return crypto.createHmac('sha256', secret).update(message).digest('hex');
}

function generateAccessToken(timestamp, apiKey, apiSecret) {
  const message = timestamp + "." + apiKey;
  const signature = crypto.createHmac('sha256', apiSecret)
                        .update(message)
                        .digest('base64');
  return signature;
}


// Widget 토큰 요청 API
app.post('/api/get-widget-token', async (req, res) => {
  try {
    const { partnerUserId, permissions } = req.body;
    
    console.log('🔑 Widget 토큰 요청:', { partnerUserId, permissions });
    
    if (!partnerUserId) {
      return res.status(400).json({
        error: 'partnerUserId is required'
      });
    }
    
    // 타임스탬프 생성 (초 단위)
    const timestamp = Math.floor(Date.now() / 1000).toString();
    
    // 요청 데이터
    const requestBody = {
      partnerUserId,
      permissions: permissions || ['DEPOSIT_READ', 'WITHDRAWAL_REQUEST', 'BALANCE_READ', 'TRANSACTION_READ']
    };
    
    // Widget API 서명 방식: METHOD + PATH + BODY + TIMESTAMP
    const method = 'POST';
    const path = '/widgets/auth/token';
    const signature = generateSignature(method, path, timestamp, PARTNER_CONFIG.apiSecret);
    
    console.log('📝 요청 데이터:', requestBody);
    console.log('🔐 생성된 서명:', signature);
    
    // Widget API에 토큰 요청
    const widgetApiResponse = await fetch('https://api.cryptoments.net/widgets/auth/token', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-API-KEY': PARTNER_CONFIG.apiKey,
        'X-TIMESTAMP': timestamp,
        'X-SIGNATURE': signature
      },
      body: JSON.stringify(requestBody)
    });
    
    if (!widgetApiResponse.ok) {
      const errorText = await widgetApiResponse.text();
      console.error('❌ Widget API 오류:', widgetApiResponse.status, errorText);
      
      return res.status(500).json({
        error: 'Failed to get widget token',
        details: errorText,
        status: widgetApiResponse.status
      });
    }
    
    const tokenData = await widgetApiResponse.json();
    console.log('✅ Widget 토큰 발급 성공');
    
    // 클라이언트에 토큰 반환
    res.json(tokenData);
    
  } catch (error) {
    console.error('❌ 토큰 발급 오류:', error);
    res.status(500).json({
      error: 'Internal server error',
      message: error.message
    });
  }
});

// 헬스체크
app.get('/health', (req, res) => {
  res.json({
    status: 'OK',
    message: 'Partner Demo Server is running',
    timestamp: new Date().toISOString()
  });
});

// 출금 요청 처리 API
app.post('/api/process-withdrawal', async (req, res) => {
  try {
    const { withdrawalData } = req.body;
    
    console.log('💸 출금 요청 받음:', withdrawalData);
    
    // 출금 데이터 검증
    if (!withdrawalData) {
      return res.status(400).json({
        error: 'withdrawalData is required'
      });
    }
    
    const { userId, currencyType, chainType, amountKrw, amountCrypto, exchangeRate, toAddress } = withdrawalData;
    
    if (!userId || !currencyType || !chainType) {
      return res.status(400).json({
        error: 'userId, currencyType, and chainType are required'
      });
    }
    
    // toAddress가 제공되지 않은 경우 기본값 사용
    const recipientAddress = toAddress || '0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6';
    
    // 금액 검증 (WithdrawalRequest.java validation 규칙에 맞춤)
    if (amountKrw !== undefined && amountCrypto !== undefined) {
      // @DecimalMin(value = "0.000001") 규칙에 맞춤
      if (amountKrw <= 0 || amountCrypto < 0.000001) {
        return res.status(400).json({
          error: 'Amount must be greater than 0.000001 (minimum token amount)'
        });
      }      
    }
    
    // 필수 필드 검증 (@NotNull 규칙에 맞춤)
    if (!currencyType || !chainType) {
      return res.status(400).json({
        error: 'currencyType and chainType are required (@NotNull validation)'
      });
    }
    
    // 출금 요청 데이터 구성 (API 문서에 맞춤)
    const withdrawalRequest = {
      partnerUserId: userId, // 파트너 사용자 ID
      amount: amountCrypto || 0, // 출금 금액 (토큰 단위)
      currencyType: currencyType, // 통화 타입 (USDT, USDC 등)
      chainType: chainType, // 체인 타입 (ETH, BSC, TRON, POLYGON)
      toAddress: recipientAddress, // 출금 대상 주소
      amountUnit: 'TOKEN', // 금액 단위 (TOKEN 또는 KRW)
      krwAmount: amountKrw || 0 // KRW 금액 (amountUnit이 KRW인 경우)
    };
    
    console.log('📝 출금 요청 데이터:', withdrawalRequest);
    
    // 타임스탬프 생성
    const timestamp = Math.floor(Date.now() / 1000).toString();
    
    // 출금 API 서명 생성
    const token = generateAccessToken(timestamp, PARTNER_CONFIG.apiKey, PARTNER_CONFIG.apiSecret);
        
    const withdrawalApiResponse = await fetch('https://api.cryptoments.net/api/v1/users/withdrawal', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-API-KEY': PARTNER_CONFIG.apiKey,
        'X-TIMESTAMP': timestamp,
        'X-ACCESS-TOKEN': token
      },
      body: JSON.stringify(withdrawalRequest)
    });
    
    if (!withdrawalApiResponse.ok) {
      const errorText = await withdrawalApiResponse.text();
      console.error('❌ 출금 API 오류:', withdrawalApiResponse.status, errorText);
      
      return res.status(500).json({
        error: 'Failed to process withdrawal',
        details: errorText,
        status: withdrawalApiResponse.status
      });
    }
    
    const withdrawalResult = await withdrawalApiResponse.json();
    console.log('✅ 출금 요청 성공:', withdrawalResult);
    
    // 클라이언트에 결과 반환
    res.json({
      success: true,
      transactionId: withdrawalResult.transactionId || `tx_${Date.now()}`,
      amount: withdrawalResult.amount || amountCrypto,
      currencyType: withdrawalResult.currencyType || currencyType,
      chainType: withdrawalResult.chainType || chainType,
      toAddress: withdrawalResult.toAddress || withdrawalRequest.toAddress,
      amountUnit: withdrawalResult.amountUnit || 'TOKEN',
      krwAmount: withdrawalResult.krwAmount || amountKrw,
      response: withdrawalResult
    });
    
  } catch (error) {
    console.error('❌ 출금 처리 오류:', error);
    res.status(500).json({
      error: 'Internal server error',
      message: error.message
    });
  }
});

// 파트너 정보 조회 (개발용)
app.get('/api/partner-info', (req, res) => {
  res.json({
    partnerId: PARTNER_CONFIG.partnerId,
    apiKey: PARTNER_CONFIG.apiKey,
    // 보안상 secret은 노출하지 않음
    hasSecret: !!PARTNER_CONFIG.apiSecret
  });
});

// 서버 시작
app.listen(PORT, () => {
  console.log(`🚀 파트너 데모 서버가 http://localhost:${PORT}에서 실행 중입니다.`);
  console.log(`📋 파트너 정보: http://localhost:${PORT}/api/partner-info`);
  console.log(`🔑 토큰 발급: POST http://localhost:${PORT}/api/get-widget-token`);
  console.log(`💸 출금 처리: POST http://localhost:${PORT}/api/process-withdrawal`);
  console.log(`🧪 v2 위젯 테스트: http://localhost:${PORT}/v2-examples`);
  console.log(`🧪 v1 위젯 테스트: http://localhost:${PORT}/usage-examples`);
  console.log(`🏥 헬스체크: http://localhost:${PORT}/health`);
});

module.exports = app;
